<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <title>TV Encarte Digital</title>
    <meta http-equiv="refresh" content="3600"> <link href="https://fonts.googleapis.com/css2?family=Montserrat:wght@400;700;900&display=swap" rel="stylesheet">
    <style>
        :root { --cor-red: #d90429; --cor-yellow: #ffb703; --cor-dark: #111; }
        body { margin: 0; overflow: hidden; background: #eee; font-family: 'Montserrat', sans-serif; }
        
        #app { width: 100vw; height: 100vh; position: relative; overflow: hidden; }

        /* --- LOADING --- */
        #loading { position: absolute; inset: 0; background: #222; color: white; display: flex; align-items: center; justify-content: center; font-size: 2em; z-index: 999; }

        /* --- ESTRUTURA GERAL DOS CARDS --- */
        .slide-container {
            display: none; width: 100%; height: 100%;
            background: radial-gradient(circle, #fff 0%, #e5e5e5 100%);
            padding: 20px; box-sizing: border-box;
            animation: fadeIn 0.8s ease;
        }
        .slide-container.active { display: grid; } /* Grid é ativado aqui */

        @keyframes fadeIn { from { opacity: 0; transform: scale(0.98); } to { opacity: 1; transform: scale(1); } }

        /* ITEM DE PRODUTO (CARD) */
        .card {
            background: white; border-radius: 15px; position: relative;
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
            display: flex; flex-direction: column; align-items: center; justify-content: center;
            overflow: hidden; border: 1px solid #ddd;
        }
        
        /* Faixa de Oferta */
        .card::before {
            content: 'OFERTA'; position: absolute; top: 15px; left: -30px;
            background: var(--cor-red); color: white; width: 120px; text-align: center;
            transform: rotate(-45deg); font-weight: bold; font-size: 14px; box-shadow: 0 2px 5px rgba(0,0,0,0.2); z-index: 5;
        }

        .prod-img { 
            width: 100%; height: 55%; object-fit: contain; 
            filter: drop-shadow(0 10px 15px rgba(0,0,0,0.2)); 
            transition: transform 0.5s;
        }
        .prod-img.multiply { mix-blend-mode: multiply; filter: contrast(1.1); }
        
        .info-area { text-align: center; width: 100%; padding: 10px; z-index: 2; }
        .prod-name { font-size: 1.8rem; font-weight: 700; color: #333; line-height: 1.1; margin-bottom: 10px; text-transform: uppercase; }
        
        /* Preço Explosão */
        .price-tag {
            background: var(--cor-yellow); color: var(--cor-red);
            display: inline-block; padding: 5px 20px;
            border-radius: 50px 10px 50px 10px;
            transform: skew(-10deg);
            box-shadow: 5px 5px 0px rgba(0,0,0,0.1);
        }
        .symbol { font-size: 1.5rem; font-weight: bold; vertical-align: super; }
        .price { font-size: 4rem; font-weight: 900; letter-spacing: -3px; }
        .validity { font-size: 0.8rem; color: #666; margin-top: 10px; }

        /* --- LAYOUTS (TEMPLATES) --- */

        /* LAYOUT 1: DESTAQUE ÚNICO (Igual ao anterior, mas grid) */
        .layout-1 { grid-template-columns: 1fr; }
        .layout-1 .card { border: none; background: transparent; box-shadow: none; flex-direction: row; }
        .layout-1 .prod-img { height: 80%; width: 50%; }
        .layout-1 .info-area { width: 50%; text-align: left; padding-left: 50px; }
        .layout-1 .prod-name { font-size: 4rem; text-shadow: 2px 2px 0px white; }
        .layout-1 .price { font-size: 9rem; }
        
        /* LAYOUT 2: COMBATE (2 Lado a Lado) */
        .layout-2 { grid-template-columns: 1fr 1fr; gap: 20px; }
        .layout-2 .prod-name { font-size: 2.2rem; }
        
        /* LAYOUT 3: ENCARTE (4 Itens) */
        .layout-4 { grid-template-columns: 1fr 1fr; grid-template-rows: 1fr 1fr; gap: 15px; }
        .layout-4 .prod-img { height: 50%; }
        .layout-4 .prod-name { font-size: 1.6rem; height: 2.4rem; overflow: hidden; } /* Limita linhas */
        .layout-4 .price { font-size: 3.5rem; }

        /* LAYOUT 4: LISTA (1 Grande + 3 Pequenos) */
        .layout-mix { 
            grid-template-columns: 1.2fr 0.8fr; 
            grid-template-rows: 1fr 1fr 1fr; 
            gap: 15px; 
            grid-template-areas: "big small1" "big small2" "big small3";
        }
        .card-big { grid-area: big; background: #fff3cd; border: 2px solid var(--cor-yellow); }
        .card-big .prod-img { height: 60%; }
        .card-big .prod-name { font-size: 3rem; }
        .card-big .price { font-size: 6rem; }
        
        /* Ajuste para telas verticais (Portrait) */
        body.portrait .layout-1 { grid-template-columns: 1fr; flex-direction: column; }
        body.portrait .layout-1 .card { flex-direction: column-reverse; }
        body.portrait .layout-2 { grid-template-columns: 1fr; grid-template-rows: 1fr 1fr; }
        body.portrait .layout-mix { 
            grid-template-columns: 1fr 1fr; 
            grid-template-rows: 1.5fr 1fr 1fr; 
            grid-template-areas: "big big" "small1 small2" "small3 small4"; /* Se tiver 5 itens */
        }

    </style>
</head>
<body class="<?php echo $_GET['modo'] ?? 'landscape'; ?>">

<div id="loading">Carregando Ofertas...</div>
<div id="app"></div>

<script>
    // Configurações
    const CONFIG = {
        modoTela: "<?php echo $_GET['modo'] ?? 'landscape'; ?>", // landscape ou portrait
        layoutFixo: "<?php echo $_GET['layout'] ?? 'mix'; ?>", // 1, 2, 4, mix (auto)
        tempoSlide: 12000 // 12 segundos por tela
    };

    const app = document.getElementById('app');
    let playlist = [];
    let gruposDeSlides = []; // Armazena os grupos já paginados
    let indiceSlide = 0;

    async function carregarDados() {
        try {
            const url = `api.php?modo=${CONFIG.modoTela}&t=${Date.now()}`;
            const resp = await fetch(url);
            const dados = await resp.json();
            
            if(dados.length > 0) {
                playlist = dados;
                processarLayouts(); // Transforma lista de produtos em lista de Telas
                document.getElementById('loading').style.display = 'none';
                if(!window.rodando) { window.rodando = true; mostrarSlide(); }
            } else {
                setTimeout(carregarDados, 10000);
            }
        } catch(e) { console.error(e); setTimeout(carregarDados, 5000); }
    }

    // A MÁGICA: Agrupa produtos em templates diferentes
    function processarLayouts() {
        gruposDeSlides = [];
        let p = [...playlist]; // Cópia para manipular
        
        // Se o usuário forçou um layout na URL (?layout=4)
        if (CONFIG.layoutFixo === '1') return criarGrupos(p, 1);
        if (CONFIG.layoutFixo === '2') return criarGrupos(p, 2);
        if (CONFIG.layoutFixo === '4') return criarGrupos(p, 4);

        // MODO MIX (Inteligente: alterna layouts para não ficar monótono)
        while(p.length > 0) {
            let sorteio = Math.random();
            
            // Lógica para decidir o próximo layout baseado em quantos produtos restam
            if (p.length >= 4 && sorteio > 0.6) {
                // Cria uma tela de 4 itens (Encarte)
                gruposDeSlides.push({ tipo: 'layout-4', itens: p.splice(0, 4) });
            } 
            else if (p.length >= 4 && sorteio > 0.3) {
                // Cria tela Mista (1 Grande + 3 Pequenos)
                gruposDeSlides.push({ tipo: 'layout-mix', itens: p.splice(0, 4) });
            }
            else if (p.length >= 2 && sorteio > 0.4) {
                // Cria tela de 2 itens (Combate)
                gruposDeSlides.push({ tipo: 'layout-2', itens: p.splice(0, 2) });
            } 
            else {
                // Sobrou 1 ou sorteio caiu aqui: Destaque Único
                gruposDeSlides.push({ tipo: 'layout-1', itens: p.splice(0, 1) });
            }
        }
    }

    function criarGrupos(lista, qtd) {
        // Função simples para layout fixo
        let classe = qtd === 1 ? 'layout-1' : (qtd === 4 ? 'layout-4' : 'layout-2');
        if (qtd === 4 && CONFIG.layoutFixo === 'mix') classe = 'layout-mix'; // Override se quiser
        
        while(lista.length > 0) {
            gruposDeSlides.push({ tipo: classe, itens: lista.splice(0, qtd) });
        }
    }

    function mostrarSlide() {
        // Limpa tela anterior
        app.innerHTML = '';

        if(indiceSlide >= gruposDeSlides.length) {
            indiceSlide = 0;
            carregarDados(); // Busca atualizações no fim do ciclo
            // Pequeno delay para garantir que o array foi refeito se tiver dados novos
            setTimeout(mostrarSlide, 1000); 
            return;
        }

        const grupo = gruposDeSlides[indiceSlide];
        
        // Cria container do slide
        const container = document.createElement('div');
        container.className = `slide-container active ${grupo.tipo}`;
        
        // Define Background (Pega do primeiro item ou usa padrão)
        const bgItem = grupo.itens[0];
        if (bgItem.bg_tipo === 'cor') container.style.background = bgItem.bg_valor;
        else if (bgItem.bg_tipo === 'imagem') container.style.background = `url('${bgItem.bg_valor}') center/cover`;
        // Se for encarte (vários itens), usa um fundo mais neutro se não tiver personalizado
        if (grupo.itens.length > 1 && bgItem.bg_tipo === 'padrao') {
            container.style.background = "#f4f4f4 url('https://www.transparenttextures.com/patterns/cubes.png')"; 
        }

        // Gera os Cards
        grupo.itens.forEach((item, idx) => {
            const card = document.createElement('div');
            // No layout Mix, o primeiro é 'card-big'
            let cardClass = 'card';
            if (grupo.tipo === 'layout-mix' && idx === 0) cardClass += ' card-big';
            
            card.className = cardClass;
            
            // Validade
            let validadeHtml = '';
            if(item.data_validade) {
                let d = item.data_validade.split('-');
                validadeHtml = `<div class="validity">Válido até ${d[2]}/${d[1]}</div>`;
            }

            // HTML do Card
            card.innerHTML = `
                <img src="${item.imagem_url}" class="prod-img ${item.efeito_visual === 'multiply' ? 'multiply' : ''}">
                <div class="info-area">
                    <div class="prod-name">${item.produto}</div>
                    <div class="price-tag">
                        <span class="symbol">R$</span>
                        <span class="price">${parseFloat(item.preco).toFixed(2).replace('.', ',')}</span>
                    </div>
                    ${validadeHtml}
                </div>
            `;
            container.appendChild(card);
        });

        app.appendChild(container);

        // Agendar próximo
        indiceSlide++;
        setTimeout(mostrarSlide, CONFIG.tempoSlide);
    }

    // Start
    carregarDados();

</script>
</body>
</html>